//
//  MP3Header.m
//  ID3
//
//  Created by Chris Drew on Tue Jul 22 2003.
//  Copyright (c) Chris Drew 2003
//

/* This class is based on MP3Header from
 * ID3.framework v0.9 by Chris Drew
 * Modified by Richard Low to contain only
 * code to work out the length of an
 * MP3 file
 */

#import "MP3Header.h"

#define LENGTHBUFFER 4*2048

// defines for MPEG Stream Type
#define MPEG1 1
#define MPEG2 2
#define MPEG25 3
#define RESERVED 0

// defines for MPEG audio Layer
#define LAYERIII 3
#define LAYERII  2
#define LAYERI   1

// defines for channel types
#define STEREO 0
#define JOINTSTEREO 1
#define DUALCHANNEL 2
#define SINGLECHANNEL 3

// declare the private methods
@interface MP3Header (PrivateAPI)
-(BOOL)findHeader;
-(BOOL)decodeHeader;
@end

@implementation MP3Header

- (void)dealloc
{
	[buffer release];
	[super dealloc];
}

- (unsigned)length:(NSString *)File
{
  BOOL headerFound = NO;
  startFrame = 0;
	int EndOfTagInFile = 0;
	bufferOffsetInFile = EndOfTagInFile;

	// open the file handle for the specified path
  NSFileHandle * file = [NSFileHandle fileHandleForReadingAtPath:File];
  if (file == NULL)
  {
		NSLog(@"Can not open file :%@", File);
    return 0;
	}
    
  // checks that MPEG file is long enough to process.  I assume that you always want to do the hash so the file need to be long enough to get sufficient data for the hash.  Currently the hash requires ~ 3*2048 Buffer is actually 4 * 2048 this leaves some space for the search (don't have to reload the buffer to hash if part the way through the buffer) and takes the buffer to whole number of cluster sizes.  So reading should be about the same speed as 3*2048.
    
  fileSize = [file seekToEndOfFile];
  if (fileSize < EndOfTagInFile + 2*LENGTHBUFFER)
  {
    NSLog(@"MP3 header processor file %@ to short to process \n\r", File);
    [file closeFile];
    return 0;
  }
    
  // get block of data after the tag in the file
  [file seekToFileOffset:EndOfTagInFile];
  buffer = [[file readDataOfLength:LENGTHBUFFER] retain];
  if (buffer == NULL) 
  {
    NSLog(@"MP3 header processor could not load data from file %@\n\r", File);
    [file closeFile];
    return 0;
  }
    
  if (!(headerFound = [self findHeader]))
  {  // if findHeader can find a header load the buffer with another chunck of data and see if we are luck a second time
    [buffer release];
    [file seekToFileOffset:EndOfTagInFile+LENGTHBUFFER-1];
    startFrame += LENGTHBUFFER-1;
    bufferOffsetInFile += LENGTHBUFFER-1;
    buffer = [[file readDataOfLength:LENGTHBUFFER] retain];
    if (buffer == NULL) 
    {
      NSLog(@"MP3 header processor could not load data from file %@\n\r",File);
      [file closeFile];
      return 0;
    }
       
    if (!(headerFound = [self findHeader]))
    {  // If you can't find a valid header a second time give it up
      NSLog(@"MP3 header processor could not find a valid MPEG header in frame on two seperate tries, file name: %@\n\r", File);
      [file closeFile];
      [buffer release];
      return 0;
    }
  }
    
  // process header
  [self decodeHeader];
    
  [file closeFile];
  return seconds;
}

-(BOOL)findHeader
{ // looks for a MPEG audio header in the file and sets the internal counter startFrame to its location in the file
    int i;
    unsigned char * charPtr = (unsigned char *) [buffer bytes];
    for (i = 0; i < LENGTHBUFFER - 1 ; i++)
    {	// looks for frame header which is two bytes 11111111 111????? 
        if ((charPtr[i] == 0xff)&&((charPtr[i+1] & 0xE0)))
        { // if is finds the two byes it sets the start frame position to the locaton of the frame within the file.  I should actually check that this is a valid frame by checking that the next frame exists.  I will do this in the future.
            startFrame += i;
            return YES;
        }
    }
    return NO;
}

-(BOOL)decodeHeader
{
/*	4th and 5th bit second byte 
        MPEG Audio version ID
           00 (0) - MPEG Version 2.5 (unofficial)
           01 (8) - reserved
           10 (10) - MPEG Version 2 (ISO/IEC 13818-3)
           11 (18) - MPEG Version 1 (ISO/IEC 11172-3)
*/
    unsigned char * charPtr = (unsigned char *) [buffer bytes];
    charPtr += startFrame - bufferOffsetInFile;

    int mpegCoding[] = { MPEG25, RESERVED, MPEG2, MPEG1 };    
    version = mpegCoding[(charPtr[1] & 0x18) >> 3];
    
/*	Layer description
           00 (0) - reserved
           01 (8) - Layer III
           10 (10) - Layer II
           11 (11) - Layer I
*/
    int layerCoding[] = { RESERVED, LAYERIII, LAYERII, LAYERI };
    layer = layerCoding[(charPtr[1] & 0x06) >> 1];

    const int bitRatesMPEG1[3][16] = 
    {	// MPEG 1
        {0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448, -1},
        {0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384, -1},
        {0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, -1}
    };
    const int bitRatesMPEG2[3][16] =
    {	// MPEG2 & 2.5
        {0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256, -1},
        {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, -1},
        {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, -1}
    };
        
    int bitRateCode = (charPtr[2] >> 4);
    if ((version == RESERVED)||(layer == RESERVED)) bitRate = 0;
    else 
    {
        if (version == MPEG1) bitRate = bitRatesMPEG1[layer-1][bitRateCode];
        else bitRate = bitRatesMPEG2[layer-1][bitRateCode];
    }
    
// frequency decode tables 
    
    const int frequencyTables[3][4] =
    { // MPEG1, MPEG2, MPEG3
        { 44100, 48000, 32000, RESERVED},
        { 22050, 24000, 16000, RESERVED},
        { 11025, 12000, 8000, RESERVED}
    };
    
    if (version == RESERVED) frequency = RESERVED;
    else frequency = frequencyTables[version-1][(charPtr[2] >>2)&3];
    
/*        Channel Mode
           00 - Stereo
           01 - Joint stereo (Stereo)
           10 - Dual channel (2 mono channels)
           11 - Single channel (Mono)  */

    const int channelTable[] = { STEREO, JOINTSTEREO, DUALCHANNEL, SINGLECHANNEL};
    channels = channelTable[charPtr[3] && 3];
    
    if (bitRate > 0)
    {
        seconds = (fileSize - startFrame)*8/(bitRate*1000);
    }
    else seconds = 0;

    
    if (version == MPEG1)
    {
        if (channels == SINGLECHANNEL) charPtr += 21;
        else charPtr += 36;
    } else
    {
        if (channels == SINGLECHANNEL) charPtr += 14;
        else charPtr += 17;
    }
    
    if ((*charPtr == 'X')&&(charPtr[1] == 'i')&&(charPtr[2] == 'n')&&(charPtr[3] == 'g'))
    { // found XING header
        XINGHeaderFound = YES;
        if ((charPtr[7] & 1))
        {
            numberOfFrames = charPtr[8]*256*256*256 + charPtr[9]*256*256 + charPtr[10]*256 +charPtr[11];
            seconds = numberOfFrames * 26 /994;
            bitRate = (500+(fileSize - startFrame)*994*8/(numberOfFrames*26))/1000;
        }
    }
    return YES;
}
@end
