//
//  MyTrackTab.m
//  XNJB
//
//  Created by Richard Low on 04/09/2004.
//

/* This class is for a tab which has an
 * NSTableView used to show items on the
 * NJB e.g. tracks or datafiles. It allows
 * sorting and searching of the list.
 */

#import "MyTableTab.h"

@implementation MyTableTab

// init/dealloc methods

- (void)dealloc
{
	[fullItemArray release];
	[truncatedItemArray release];
	[super dealloc];
}

/**********************/

/* to be implemented by subclass:
 * fill the njbTable
 */
- (id)tableView:(NSTableView *)aTableView
   objectValueForTableColumn:(NSTableColumn *)aTableColumn
						row:(int)rowIndex
{
  return nil;	
}

/* to be implemented by subclass:
 * search the njbTable for strings
 * containing the contents of njbSearchField
 * called when the search box has been modified
 * or we fullItemArray has been changed
 */
- (void)searchNJBTable{}

/* clears the sorting arrows on the table
 * column headers
 */
- (void)clearSortingArrows
{
	// clear any sorting arrows as we won't sort the incoming data here
	NSArray *columns = [njbTable tableColumns];
	int i = -1;
	for (i = 0; i < [columns count]; i++)
	{
		[njbTable setIndicatorImage:nil inTableColumn:[columns objectAtIndex:i]];
	}
	
	// reset memory of last column clicked on
	currentColumnID = nil;
}

/* the user has clicked on a table column
 * header so sort the table appropriately
 */
- (void)tableView:(NSTableView *)tableView didClickTableColumn:(NSTableColumn *)tableColumn
{	
	if (tableView != njbTable)
		return;
	
	NSString *ident = [tableColumn identifier];
	
	sortDirections sortDirection = SORT_ASCENDING;
	// currentColumnID is the column last selected
	// see if we've clicked on it again to work
	// out the sort direction
	if (ident == currentColumnID)
	{
		switch (currentSortDirection)
		{
			case SORT_ASCENDING:
				sortDirection = SORT_DESCENDING;
				break;
			case SORT_DESCENDING:
				sortDirection = SORT_ASCENDING;
				break;
			default:
				NSLog(@"Invalid currentSortDirection in MyFileTab!");
				break;
		}
	}
	else
		// default sort direction
		sortDirection = SORT_ASCENDING;
	
	SEL newSortSelector = [self sortSelectorForColumn:ident];
	if (currentColumnID != nil)
	{
		// if the column has changed, then update the oldSortSelector, otherwise remember which column was previously clicked
		if (currentColumnID != ident)
			oldSortSelector = [self sortSelectorForColumn:currentColumnID];
		// sort both arrays
		[fullItemArray sortUsingSelector:newSortSelector andSelector:oldSortSelector inDirection:sortDirection];
		[truncatedItemArray sortUsingSelector:newSortSelector andSelector:oldSortSelector inDirection:sortDirection];
	}
	else
	{
		// sort both arrays with no secondary sort selector
		[fullItemArray sortUsingSelector:newSortSelector inDirection:sortDirection];
		[truncatedItemArray sortUsingSelector:newSortSelector inDirection:sortDirection];
	}
	
	// set the column sort arrow
	switch (sortDirection)
	{
		case SORT_ASCENDING:
			[tableView setIndicatorImage:[NSImage imageNamed:@"NSAscendingSortIndicator"] inTableColumn:tableColumn];
			break;
		case SORT_DESCENDING:
			[tableView setIndicatorImage:[NSImage imageNamed: @"NSDescendingSortIndicator"] inTableColumn:tableColumn];
			break;
		default:
			NSLog(@"Invalid sortDirection in MyFileTab!");
			break;
	}
	
	// clear arrow on previous selected column
	if (currentColumnID != ident)
		[tableView setIndicatorImage:nil inTableColumn:[tableView tableColumnWithIdentifier:currentColumnID]];
	
	currentColumnID = ident;
	currentSortDirection = sortDirection;	
	
	[njbTable reloadData];
}

/* returns the array currently displaying
 * truncatedItemArray is the array of items that 
 * match the search box. fullItemArray is all items.
 */
- (MyMutableArray *)itemArrayDisplaying
{
	if (truncatedItemArray != nil)
		return truncatedItemArray;
	else
		return fullItemArray;
}

- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	if (aTableView != njbTable)
		return 0;
	if (fullItemArray == nil)
		return 0;
	return [[self itemArrayDisplaying] count];
}

/* to be implemented by subclass:
 * returns the sort selector for the column with
 * ID columnID
 */
- (SEL)sortSelectorForColumn:(NSString *)columnID
{
	return nil;
}

/* called when we connect to an NJB
 * clear the outdated arrays
 */
- (void)NJBConnected:(NSNotification *)note
{
	// clear the list if there was one so we can't change stuff while getting new file list
	[fullItemArray release];
	fullItemArray = nil;
	[truncatedItemArray release];
	truncatedItemArray = nil;
	[njbTable reloadData];
	
	// this must be done last as we might set fullItemArray in onConnectAndActive
	[super NJBConnected:note];
}

- (MyMutableArray *)fullItemArray
{
	return fullItemArray;
}

/* delegate method called when
 * the search box is modified
 */
- (IBAction)searchNJBTableAction:(id)sender
{
	// njbSearchField has been changed
	[self searchNJBTable];
}

@end
