#include <stdlib.h>
#include <string.h>
#include "libnjb.h"
#include "njb_error.h"
#include "defs.h"
#include "protocol3.h"
#include "byteorder.h"
#include "unicode.h"

#define FR_UNI_TITLE	"UNI_TITLE"
#define FR_UNI_ALBUM	"UNI_ALBUM"
#define FR_UNI_GENRE	"UNI_GENRE"
#define FR_UNI_ARTIST	"UNI_ARTIST"
/* Haven't seen this one being used, but add it anyway. */
#define FR_UNI_FNAME	"UNI_FNAME"

extern int njb_error;
extern int __sub_depth;
extern int njb_unicode_flag;

songid_t *songid_new (void)
{
	__dsub= "songid_new";
	songid_t *song;

	song= (songid_t *) malloc (sizeof(songid_t));
	if ( song == NULL ) {
		njb_error_add(__sub, EO_NOMEM);
		return NULL;
	}

	memset(song, 0, sizeof(songid_t));
	song->next = NULL;

	return song;
}

songid_t *songid_unpack (void *data, size_t nbytes)
{
	__dsub= "songid_unpack";
	unsigned char *dp= (unsigned char *) data;
	size_t index;
	songid_t *song;
	u_int16_t i, nframes;
	/* The unicode tags shall take precedence */
	int had_uni_title = 0;
	int had_uni_album = 0;
	int had_uni_genre = 0;
	int had_uni_artist = 0;
	int had_uni_fname = 0;

	song = songid_new();
	if ( song == NULL ) return NULL;

	nframes = njb1_bytes_to_16bit(&dp[0]);

	index = 2;

	/* Loop through the frames of a single song ID */
	for (i = 0; i < nframes; i++) {
	        /* Each frame has a label, a value and a type */
		u_int16_t lsize, vsize, type;
		char *label = NULL;
		char *value = NULL;
		songid_frame_t *frame = NULL;

		type = njb1_bytes_to_16bit(&dp[index]);
		lsize = njb1_bytes_to_16bit(&dp[index+2]);
		vsize = njb1_bytes_to_16bit(&dp[index+4]);
		index += 8;
		label = &dp[index];
		value = &dp[index+lsize];

		/* Handle Unicode conversion on ASCII fields, or default
		 * to ISO 8859-1 */
		if (type == ID_DATA_ASCII) {
		  if (njb_unicode_flag == NJB_UC_UTF8) {
		    if ( (!strcmp(label, FR_TITLE) && had_uni_title) ||
			 (!strcmp(label, FR_ALBUM) && had_uni_album) ||
			 (!strcmp(label, FR_GENRE) && had_uni_genre) ||
			 (!strcmp(label, FR_ARTIST) && had_uni_artist) ||
			 (!strcmp(label, FR_FNAME) && had_uni_fname) ) {
		      frame = NULL;
		    } else {
		      char *utf8str = NULL;
		      int utf8len;
		      
		      utf8str = strtoutf8(value);
		      if (utf8str == NULL) {
			songid_destroy(song);
			njb_error_add(__sub, EO_NOMEM);
			return NULL;
		      }
		      utf8len= strlen(utf8str)+1;
		      frame = songid_frame_new(lsize, label, type, utf8len,
					       utf8str);
		      free(utf8str);
		    }
		  } else {
		    /* No Unicode conversion, default by treating as ISO 8859-1 */
		    frame = songid_frame_new(lsize, label, type, strlen(value)+1,
					     value);
		  }
		/*
		 * The Unicode labels appeared in later software, and shall take
		 * precedence if present.
		 */
		} else if (type == ID_DATA_UNI) {
		  char *utf8str = NULL;
		  int utf8len;

		  utf8str = ucs2tostr(value);
		  if (utf8str == NULL) {
		    songid_destroy(song);
		    njb_error_add(__sub, EO_NOMEM);
		    return NULL;
		  }
		  utf8len = strlen(utf8str)+1;
		  /*
		   * After converting the Unicode frame into an UTF8 ASCII 
		   * frame, we use the standard labels to simplify the API.
		   */
		  if (!strcmp(label, FR_UNI_TITLE)) {
		    frame = songid_frame_new(strlen(FR_TITLE)+1, FR_TITLE, ID_DATA_ASCII, utf8len,
					     utf8str);
		    had_uni_title = 1;
		  } else if (!strcmp(label, FR_UNI_ALBUM)) {
		    frame = songid_frame_new(strlen(FR_ALBUM)+1, FR_ALBUM, ID_DATA_ASCII, utf8len,
					     utf8str);
		    had_uni_album = 1;
		  } else if (!strcmp(label, FR_UNI_GENRE)) {
		    frame = songid_frame_new(strlen(FR_GENRE)+1, FR_GENRE, ID_DATA_ASCII, utf8len,
					     utf8str);
		    had_uni_genre = 1;
		  } else if (!strcmp(label, FR_UNI_ARTIST)) {
		    frame = songid_frame_new(strlen(FR_ARTIST)+1, FR_ARTIST, ID_DATA_ASCII, utf8len,
					     utf8str);
		    had_uni_artist = 1;
		  } else if (!strcmp(label, FR_UNI_FNAME)) {
		    frame = songid_frame_new(strlen(FR_FNAME)+1, FR_FNAME, ID_DATA_ASCII, utf8len,
					     utf8str);
		    had_uni_fname = 1;
		  }
		  free(utf8str);
		} else { /* This means it is a numeric value */
		  /*
		   * FIXME: check that this can fix some endianness issues - 
		   * the values arrive in device endianness 
		   */
		  if (vsize == 2) {
		    u_int16_t *dummy = (u_int16_t *) value;
		    *dummy = njb1_bytes_to_16bit(value);
		  } else if (vsize == 4) {
		    u_int32_t *dummy = (u_int32_t *) value;
		    *dummy = njb1_bytes_to_32bit(value);
		  } else if (vsize == 8) {
		    u_int64_t *dummy = (u_int64_t *) value;
		    *dummy = njb1_bytes_to_64bit(value);
		  } else {
		    /* Just copy it, whatever it is. */
		  }
		  frame = songid_frame_new(lsize, label, type, vsize, value);
		}
		/*
		 * The frame will only be NULL if there was a Unicode frame
		 * taking precedence over the ISO 8859-1 frame.
		 */
		if (frame != NULL) {
		  songid_addframe(song, frame);
		}

		index += (lsize + vsize);

		if ( index > nbytes ) {
			songid_destroy(song);
			njb_error_add(__sub, EO_BADDATA);
			return NULL;
		}
	}

	return song;
}

unsigned char *songid_pack (songid_t *song, u_int32_t *tagsize)
{
	__dsub= "songid_pack";
	songid_frame_t *frame;
	unsigned char tagbuffer[1024];
	unsigned char *data;
	size_t index;

	*tagsize = 0;

	if ( !song->nframes ) {
	        /* The song has to have frames */
		njb_error_add(__sub, EO_BADDATA);
		return NULL;
	}

	from_16bit_to_njb1_bytes(song->nframes, &tagbuffer[0]);
	index = 2;

	songid_reset_getframe(song);
	/* FIXME: add UNI_ unicode frames for all applicable frames, except filename */
	while ((frame = songid_getframe(song))) {
	        /* Add type and size for this frame */
		from_16bit_to_njb1_bytes(frame->type, &tagbuffer[index]);
		from_16bit_to_njb1_bytes(frame->labelsz, &tagbuffer[index+2]);
		index += 4;

		/* ASCII frames such as filename, artist, album... */
		if (frame->type ==  ID_DATA_ASCII) {
		  char *frame_ascii_content = NULL;
		  u_int16_t frame_ascii_len = 0;

		  /* Convert ASCII fields to ISO 8859-1 as is used on NJB 1 */
		  if(njb_unicode_flag == NJB_UC_UTF8) {
		    frame_ascii_content = utf8tostr(frame->data);
		  } else {
		    frame_ascii_content = strdup(frame->data);
		  }
		  if (frame_ascii_content == NULL) {
		    njb_error_add(__sub, EO_NOMEM);
		    return NULL;
		  }

		  frame_ascii_len = strlen(frame_ascii_content)+1;
		  from_16bit_to_njb1_bytes(frame_ascii_len, &tagbuffer[index]);
		  tagbuffer[index+2] = 0x00;
		  tagbuffer[index+3] = 0x00;
		  index += 4;
		  memcpy(&tagbuffer[index], frame->label, frame->labelsz);
		  index += frame->labelsz;
		  memcpy(&tagbuffer[index], frame_ascii_content, frame_ascii_len);
		  free(frame_ascii_content);
		  index += frame_ascii_len;

		} else {
		  /* Numerical tags such as year or length */
		  from_16bit_to_njb1_bytes(frame->datasz, &tagbuffer[index]);
		  tagbuffer[index+2] = 0x00;
		  tagbuffer[index+3] = 0x00;
		  index += 4;
		  memcpy(&tagbuffer[index], frame->label, frame->labelsz);
		  index += frame->labelsz;
		  if ( frame->datasz == 1 ) {
		    unsigned char *dp = (unsigned char *) frame->data;

		    tagbuffer[index] = *dp;
		  } else if ( frame->datasz == 2 ) {
		    from_16bit_to_njb1_bytes(*(u_int16_t *)frame->data, 
					     &tagbuffer[index]);
		  } else if ( frame->datasz == 4 ) {
		    from_32bit_to_njb1_bytes(*(u_int32_t *)frame->data,
					     &tagbuffer[index]);
		  } else if ( frame->datasz == 8) {
		    from_64bit_to_njb1_bytes(*(u_int64_t *)frame->data,
					     &tagbuffer[index]);
		  } else {
		    /* Only numerical frames of size 1,2,4 and 8 are allowed. */
		    njb_error_add(__sub, EO_BADDATA);
		    return NULL;
		  }
		  index += frame->datasz;
		}
	}

	*tagsize = index;
	if (*tagsize == 0) {
		njb_error_add(__sub, EO_BADDATA);
		return NULL;
	}
	/* Duplicate the buffer and return */
	data = (unsigned char *) malloc (*tagsize);
	if ( data == NULL ) {
		njb_error_add(__sub, EO_NOMEM);
		return NULL;
	}
	memcpy(data, tagbuffer, *tagsize);

	return data;
}

/* These functions deal with putting data in packed NJB3 tags */
static u_int32_t valtoint32(unsigned char *data, int valsz)
{
  u_int32_t retval = 0;

  if (valsz == 2) {
    u_int16_t dummy;
    u_int16_t *temp = (u_int16_t *) data;

    /* retval = (data[1]<<8) + data[0]; - old bad endianness code */
    dummy = *temp;
    retval = (u_int32_t) dummy;
  }
  else if (valsz == 4) {
    u_int32_t *temp = (u_int32_t *) data;
    /* retval = (data[3]<<24) + (data[2]<<16) + (data[1]<<8) + data[0]; - old bad endianess code */
    retval = *temp;
  }
  return retval;
}

static void add_bin_unistr(unsigned char *data, u_int32_t *datap, u_int32_t tagtype, unsigned char *unistr)
{
  u_int32_t binlen;

  binlen = ucs2strlen(unistr) * 2 + 2;
  from_16bit_to_njb3_bytes(binlen+2, &data[*datap]);
  *datap += 2;
  from_16bit_to_njb3_bytes(tagtype, &data[*datap]);
  *datap += 2;
  memcpy(data+(*datap), unistr, binlen);
  *datap += binlen;
}

/* Same thing but for NJB3 */
unsigned char *songid_pack3 (songid_t *song, u_int32_t *tagsize)
{
	__dsub= "songid_pack3";
	songid_frame_t *frame;
	unsigned char *data = NULL;
	u_int32_t datap = 0;
	u_int32_t track_size = 0;
	u_int16_t track_year = 0;
	u_int32_t track_number = 0;
	u_int32_t track_length = 0;
	u_int32_t track_protected = 0;
	u_int32_t track_codec = NJB3_CODEC_MP3_ID;
	unsigned char *track_title = NULL;
	unsigned char *track_album = NULL;
	unsigned char *track_artist = NULL;
	unsigned char *track_genre = NULL;
	unsigned char *track_fname = NULL;

	/* Two bytes padding at the end plus
	 * ever-present tags (year, track number, codec) */
	*tagsize = 20;

	if ( !song->nframes ) {
		njb_error_add(__sub, EO_BADDATA);
		return NULL;
	}

	songid_reset_getframe(song);

	while ( (frame= songid_getframe(song)) ) {
	  if (!strcmp(frame->label, FR_SIZE)) {
	    /* 8 bytes, 2 length 2 tagid 4 bytes size */
	    track_size = valtoint32(frame->data, frame->datasz);
	    *tagsize+= 8;
	  }
	  else if (!strcmp(frame->label, FR_YEAR)) {
	    /* 6 bytes, 2 length 2 tagid 2bytes year */
	    /* This is a string so first convert it to an integer */
	    /* The interface uses a string, but PROTOCOL3 series
	     * uses an u_int32 value, so convert it */
	    char *dummy;
	    track_year = (u_int16_t) strtoul(frame->data, &dummy, 10);
	  }
	  else if (!strcmp(frame->label, FR_TRACK)) {
	    /* 6 bytes, 2 length 2 tagid 2bytes trackno */
	    track_number = valtoint32(frame->data, frame->datasz);
	  }
	  else if (!strcmp(frame->label, FR_LENGTH)) {
	    /* 6 bytes, 2 length 2 tagid 2bytes length */
	    track_length = valtoint32(frame->data, frame->datasz);
	    *tagsize += 6;
	  }
	  /* For strings, stringlengh * 2 + 4 bytes */
	  else if (!strcmp(frame->label, FR_TITLE)) {
	    track_title = strtoucs2(frame->data);
	    *tagsize += (6 + 2*ucs2strlen(track_title));
	  }
	  else if (!strcmp(frame->label, FR_ALBUM)) {
	    track_album = strtoucs2(frame->data);
	    *tagsize += (6 + 2*ucs2strlen(track_album));
	  }
	  else if (!strcmp(frame->label, FR_ARTIST)) {
	    track_artist = strtoucs2(frame->data);
	    *tagsize += (6 + 2*ucs2strlen(track_artist));
	  }
	  else if (!strcmp(frame->label, FR_GENRE)) {
	    track_genre = strtoucs2(frame->data);
	    *tagsize += (6 + 2*ucs2strlen(track_genre));
	  }
	  else if (!strcmp(frame->label, FR_FNAME)) {
	    track_fname = strtoucs2(frame->data);
	    *tagsize += (6 + 2*ucs2strlen(track_fname));
	  }
	  else if (!strcmp(frame->label, FR_CODEC)) {
	    /* 6 bytes, 2 length 2 tagid 2bytes year */
	    if (!strcmp(frame->data, NJB_CODEC_MP3)) {
	      /* Use old codec name NJB3_CODEC_MP3_ID_OLD on old firmware? */
	      track_codec = NJB3_CODEC_MP3_ID;
	    } else if (!strcmp(frame->data, NJB_CODEC_WAV)) {
	      track_codec = NJB3_CODEC_WAV_ID;
	    } else if (!strcmp(frame->data, NJB_CODEC_WMA)) {
	      track_codec = NJB3_CODEC_WMA_ID;
	    }
	  }
	  else if (!strcmp(frame->label, FR_PROTECTED)) {
	    track_protected = 1;
	  }
	}

	/* No extra information at all? Bad. */
	if ( *tagsize == 20) {
		njb_error_add(__sub, EO_BADDATA);
		return NULL;
	}

	data= (unsigned char *) malloc (*tagsize);
	if ( data == NULL ) {
		njb_error_add(__sub, EO_NOMEM);
		return NULL;
	}
	memset(data, 0, *tagsize);

	/* Then add the album strings */
	if (track_title != NULL)
	  add_bin_unistr(data, &datap, NJB3_TITLE_FRAME_ID, track_title);
	if (track_album != NULL)
	  add_bin_unistr(data, &datap, NJB3_ALBUM_FRAME_ID, track_album);
	if (track_artist != NULL)
	  add_bin_unistr(data, &datap, NJB3_ARTIST_FRAME_ID, track_artist);
	if (track_genre != NULL)
	  add_bin_unistr(data, &datap, NJB3_GENRE_FRAME_ID, track_genre);
	if (track_size != 0) {
	  from_16bit_to_njb3_bytes(6, &data[datap]);
	  datap += 2;
	  from_16bit_to_njb3_bytes(NJB3_FILESIZE_FRAME_ID, &data[datap]);
	  datap += 2;
	  from_32bit_to_njb3_bytes(track_size, &data[datap]);
	  datap += 4;
	}
	/* Default to unlocked files if == 0 */
	if (track_protected != 0) {
	  from_16bit_to_njb3_bytes(4, &data[datap]);
	  datap += 2;
	  from_16bit_to_njb3_bytes(NJB3_FILESIZE_FRAME_ID, &data[datap]);
	  datap += 2;
	  from_16bit_to_njb3_bytes(1, &data[datap]);
	  datap += 2;
	}
	/* These three are always stored */
	from_16bit_to_njb3_bytes(4, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(NJB3_CODEC_FRAME_ID, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(track_codec, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(4, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(NJB3_YEAR_FRAME_ID, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(track_year, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(4, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(NJB3_TRACKNO_FRAME_ID, &data[datap]);
	datap += 2;
	from_16bit_to_njb3_bytes(track_number, &data[datap]);
	datap += 2;
	if (track_length != 0) {
	  from_16bit_to_njb3_bytes(4, &data[datap]);
	  datap += 2;
	  from_16bit_to_njb3_bytes(NJB3_LENGTH_FRAME_ID, &data[datap]);
	  datap += 2;
	  from_16bit_to_njb3_bytes(track_length, &data[datap]);
	  datap += 2;
	}
	if (track_fname != NULL)
	  add_bin_unistr(data, &datap, NJB3_FNAME_FRAME_ID, track_fname);

	if (track_title != NULL)
	  free(track_title);
	if (track_album != NULL)
	  free(track_album);
	if (track_artist != NULL)
	  free(track_artist);
	if (track_genre != NULL)
	  free(track_genre);
	if (track_fname != NULL)
	  free(track_fname);
	return data;
}


void songid_addframe (songid_t *song, songid_frame_t *frame)
{
	if ( song->nframes ) {
		song->last->next= frame;
		song->last= frame;
	} else {
		song->first= song->last= frame;
		song->cur= song->first;
	}

	frame->next= NULL;
	song->nframes++;
}

void songid_destroy (songid_t *song)
{
	songid_frame_t *frame;

	songid_reset_getframe(song);
	while ( (frame= songid_getframe(song)) ) {
		songid_frame_destroy(frame);
	}

	free(song);
}

void songid_reset_getframe (songid_t *song)
{
	song->cur= song->first;
}

songid_frame_t *songid_getframe (songid_t *song)
{
	songid_frame_t *t;

	if ( song->cur == NULL ) return NULL;
	t= song->cur;
	song->cur= song->cur->next;

	return t;
}

void songid_dump (songid_t *song, FILE *fp)
{
	songid_frame_t *frame;

	songid_reset_getframe(song);
	fprintf(fp, "ID: %u\n", song->trid);
	while( (frame= songid_getframe(song)) ) {
		songid_frame_dump(frame, fp);
	}
}

songid_frame_t *songid_findframe (songid_t *song, const char *label)
{
	songid_frame_t *frame;

	songid_reset_getframe(song);
	while ( (frame= songid_getframe(song)) ) {
		if ( ! strcmp(frame->label, label) ) return frame;
	}

	return NULL;
}

/*
 * Song ID frame methods 
 */

songid_frame_t *songid_frame_new (u_int16_t labelsz, const void *label,
	u_int16_t type, u_int16_t datasz, const void *data)
{
	__dsub= "songid_frame_new";
	songid_frame_t *frame;

	frame= (songid_frame_t *) malloc(sizeof(songid_frame_t));
	if ( frame == NULL ) {
		return NULL;
		njb_error_add(__sub, EO_NOMEM);
	}

	frame->label= malloc(labelsz);
	frame->data= malloc(datasz);
	if ( frame->label == NULL || frame->data == NULL ) {
		songid_frame_destroy(frame);
		njb_error_add(__sub, EO_NOMEM);
		return NULL;
	}

	frame->labelsz= labelsz;
	frame->datasz= datasz;
	frame->type= type;
	memcpy(frame->label, label, labelsz);
	memcpy(frame->data, data, datasz);

	return frame;
}

void songid_frame_destroy (songid_frame_t *frame)
{
	if ( frame->label != NULL ) free(frame->label);
	if ( frame->data != NULL ) free(frame->data);
	free(frame);
}

void songid_frame_dump (songid_frame_t *frame, FILE *fp)
{
	char *label;

	label= (char *) malloc (frame->labelsz + 1);
	if ( label == NULL ) {
		fprintf(fp, "(malloc failure)\n");
		return;
	}
	memcpy(label, frame->label, frame->labelsz);

	fprintf(fp, "%s: ", label);
	free(label);

	if ( frame->type == ID_DATA_ASCII ) {
		char *data;
		data= (char *) malloc (frame->datasz + 1);
		if ( data == NULL ) {
			fprintf(fp, "(malloc failure)\n");
			return;
		}
		memcpy(data, frame->data, frame->datasz);
		fprintf(fp, "%s\n", data);
		free(data);
	} else {
		if ( frame->datasz > 4 ) {
			fprintf(fp, "(can't display 64-bit int)\n");
		} else {
			fprintf(fp, "%u\n", songid_frame_data32(frame));
		}
	}
}

u_int32_t songid_frame_data32 (songid_frame_t *frame)
{
	if ( frame->datasz == 1 ) {
		char tval = ((char *) frame->data)[0];
		return (u_int32_t) tval;
	} else if ( frame->datasz == 2 ) {
		u_int16_t tval;
		memcpy(&tval, frame->data, 2);
		return (u_int32_t) tval;
	} else if ( frame->datasz == 4 ) {
		u_int32_t tval;
		memcpy(&tval, frame->data, 4);
		return tval;
	} 

	return 0;
}

/*
 * Info methods
 */

u_int32_t songid_size (songid_t *song)
{
	songid_frame_t *frame;

	frame= songid_findframe(song, FR_SIZE);
	if ( frame == NULL ) return 0;

	return (u_int32_t) songid_frame_data32(frame);
}

u_int32_t songid_length (songid_t *song)
{
	songid_frame_t *frame;

	frame= songid_findframe(song, FR_LENGTH);
	if ( frame == NULL ) return 0;

	return (u_int32_t) songid_frame_data32(frame);
}

u_int32_t songid_track (songid_t *song)
{
	songid_frame_t *frame;

	frame= songid_findframe(song, FR_TRACK);
	if ( frame == NULL ) return 0;

	return (u_int32_t) songid_frame_data32(frame);
}

const char *songid_label (songid_t *song, const char *label)
{
	songid_frame_t *frame;

	frame= songid_findframe(song, label);
	if ( frame == NULL ) return NULL;

	return (const char *) frame->data;
}
